/*=====================================================================
Any authorized distribution of any copy of this code (including any related documentation) must reproduce the following restrictions, disclaimer and copyright notice:
The Genesys name, trademarks and/or logo(s) of Genesys shall not be used to name (even as a part of another name), endorse and/or promote products derived from this code without prior written permission from Genesys Telecommunications Laboratories, Inc.
The use, copy, and/or distribution of this code is subject to the terms of the Genesys Developer License Agreement.  This code shall not be used, copied, and/or distributed under any other license agreement.
THIS CODE IS PROVIDED BY GENESYS TELECOMMUNICATIONS LABORATORIES, INC. (GENESYS) AS IS WITHOUT ANY WARRANTY OF ANY KIND. GENESYS HEREBY DISCLAIMS ALL EXPRESS, IMPLIED, OR STATUTORY CONDITIONS, REPRESENTATIONS AND WARRANTIES WITH RESPECT TO THIS CODE (OR ANY PART THEREOF), INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. GENESYS AND ITS SUPPLIERS SHALL NOT BE LIABLE FOR ANY DAMAGE SUFFERED AS A RESULT OF USING THIS CODE. IN NO EVENT SHALL GENESYS AND ITS SUPPLIERS BE LIABLE FOR ANY DIRECT, INDIRECT, CONSEQUENTIAL, ECONOMIC, INCIDENTAL, OR SPECIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, ANY LOST REVENUES OR PROFITS).
Copyright &copy; 2010 Genesys Telecommunications Laboratories, Inc. All rights reserved.
=====================================================================*/
package com.genesyslab.samples.interceptor;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.util.Date;
import java.text.SimpleDateFormat;

/**
 * This is an helper class to be used by the custom interceptor objects.
 * It maintains (open, append, close) a plain text file used for audit trail logging.
 * This class implements a singleton object that can be accessed from everywhere in the java application.
 * All the methods of this singleton are synchronized to avoid concurrency while accessing file resource.
 * In case of high load or if the plain text file is replaced by a slower data source it could be
 * recommended to decouple the data source access from the calls to the manager methods 
 * using a task queue and a dedicated thread for example.
 */
public class AuditFileManager {

    private BufferedOutputStream logBuffer;
    
	private static AuditFileManager instance;
	private static SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd_HHmmss");
	
    /**
     * Singleton accessor
     */
	public static AuditFileManager getInstance() {
        if (instance != null)
            return instance;
        instance = new AuditFileManager();

        return instance;
	}
	
    /**
     * Open an audit log file in the specified directory.
     * Audit file name is based on a time stamp
     * @param rootPath The root directory of the audit log file
     */
	public synchronized void open(String rootPath) {
        if (logBuffer==null) {
            Date now = new Date();
            StringBuffer fileName = new StringBuffer(rootPath);
            fileName.append(File.separator);
            fileName.append("audit_");
            fileName.append( dateFormat.format(now) );
            fileName.append(".log");
            File file = new File(fileName.toString());
            try {
                FileOutputStream fos = new FileOutputStream(file);
                logBuffer = new BufferedOutputStream(fos);
            } catch (Exception e) {
                System.out.println("file opening failed:"+e);
            }
        } else {
            System.out.println("There is already an opened audit file");
        }
	}
    
    /**
     * Append a message to the current open directory
     * @param message The log message that must be appended to the audit file
     */
    public synchronized void append(String message) {
        if (logBuffer!=null) {
            Date now = new Date();
            String line = now.toString()+":"+message+"\n";
            try {
                logBuffer.write(line.getBytes());
                logBuffer.flush();
            } catch (Exception e) {
                System.out.println("write in file failed:"+e);
            }
        } else {
            System.out.println("There isn't any opened audit file");
        }
    }
    
    /**
     * Close the current audit file
     */
    public synchronized void close() {
        if (logBuffer!=null) {
            try {
                logBuffer.close();
                logBuffer = null;
            } catch (Exception e) {
                System.out.println("file closure failed:"+e);
            }
        }
    }
}
